package demo.server_push.websocket;

import org.zkoss.bind.BindUtils;
import org.zkoss.bind.annotation.*;
import org.zkoss.zk.ui.Desktop;
import org.zkoss.zk.ui.Executions;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

public class RobotTrackViewModel {
	private static final String RUNNING = "running";
	public static final String POSITION = "position";
	public static final String INTERVAL = "interval";
	public static final String UPDATE_INTERVAL = "updateInterval";

	private Coord position = new Coord(50, 50);
	private AtomicBoolean running = new AtomicBoolean(false);
	private int interval = 500;

	@Command
	@NotifyChange(RUNNING)
	public void start(@ContextParam(ContextType.DESKTOP) Desktop desktop) {
		updatePosition();
		this.running.set(true);
		startBackgroundThread(desktop);
	}

	@Command
	@NotifyChange(RUNNING)
	public void stop(@ContextParam(ContextType.DESKTOP) Desktop desktop) {
		this.running.set(false);
	}

	@GlobalCommand(UPDATE_INTERVAL)
	@NotifyChange(INTERVAL)
	public void updateInterval(@BindingParam("interval") int interval) {
		this.interval = Math.min(Math.max(interval, 200), 2000);
	}

	private void startBackgroundThread(Desktop desktop) {
		new Thread(() -> {
			while (running.get()) {
				try {
					activated(desktop, () -> {
						this.updatePosition();
					});
					Thread.sleep(interval);
				} catch(Exception e) {
					//leave background thread, no error handling in this Demo
					return;
				}
			}
		}).start();
	}

	private void activated(Desktop desktop, Runnable task) {
		try {
			Executions.activate(desktop);
			task.run();
		} catch (Exception e) {
			throw new RuntimeException(e);
		} finally {
			Executions.deactivate(desktop);
		}
	}

	private void updatePosition() {
		double now = (double) System.nanoTime() / TimeUnit.SECONDS.toNanos(1);
		this.position = new Coord(Math.cos(now / Math.PI / 2.0) * 40 + 50, Math.sin(now / Math.PI / 2.0) * 40 + 50);
		BindUtils.postNotifyChange(null, null, this, POSITION);
	}

	public boolean getRunning() {
		return running.get();
	}

	public Coord getPosition() {
		return position;
	}
}
